<?php
/**
 * Plugin Name: Image Hover Add-on for Fusion Builder
 * Plugin URI: https://fbimagehover.com/
 * Description: Image Hover add-on for Fusion Builder adds 200+ different image hover styles.
 * Version: 1.0
 * Author: InfiWebs
 * Author URI: https://www.infiwebs.com
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Plugin version.
if ( ! defined( 'FB_IMAGE_HOVER_VERSION' ) ) {
	define( 'FB_IMAGE_HOVER_VERSION', '1.0' );
}
// Plugin Root File.
if ( ! defined( 'FB_IMAGE_HOVER_PLUGIN_FILE' ) ) {
	define( 'FB_IMAGE_HOVER_PLUGIN_FILE', __FILE__ );
}
// Plugin Folder Path.
if ( ! defined( 'FB_IMAGE_HOVER_PLUGIN_DIR' ) ) {
	define( 'FB_IMAGE_HOVER_PLUGIN_DIR', wp_normalize_path( plugin_dir_path( FB_IMAGE_HOVER_PLUGIN_FILE ) ) );
}
// Plugin Folder URL.
if ( ! defined( 'FB_IMAGE_HOVER_PLUGIN_URL' ) ) {
	define( 'FB_IMAGE_HOVER_PLUGIN_URL', plugin_dir_url( FB_IMAGE_HOVER_PLUGIN_FILE ) );
}

if ( ! class_exists( 'Fusion_Builder_Image_Hover' ) ) {

	/**
	 * Main Fusion_Builder_Image_Hover Class.
	 *
	 * @since 1.0
	 */
	class Fusion_Builder_Image_Hover {

		/**
		 * The one, true instance of this object.
		 *
		 * @since 1.0
		 * @static
		 * @access private
		 * @var object
		 */
		private static $instance;

		/**
		 * Creates or returns an instance of this class.
		 *
		 * @since 1.0
		 * @static
		 * @access public
		 */
		public static function get_instance() {

			// If an instance hasn't been created and set to $instance create an instance and set it to $instance.
			if ( null === self::$instance ) {
				self::$instance = new Fusion_Builder_Image_Hover();
			}
			return self::$instance;
		}

		/**
		 * Initializes the plugin by setting localization, hooks, filters,
		 * and administrative functions.
		 *
		 * @since 1.0
		 * @access private
		 */
		private function __construct() {

			// Add admin notice if Fusion Builder is deactivated or not installed.
			add_action( 'admin_notices', array( $this, 'fusion_builder_required_admin_notice' ) );

			// Load plugin textdomain.
			add_action( 'plugins_loaded', array( $this, 'textdomain' ) );

			// Enqueue styles on frontend.
			add_action( 'wp_enqueue_scripts', array( $this, 'frontend_styles' ), 9 );

			// Add front-end templates.
			if ( function_exists( 'fusion_is_builder_frame' ) && fusion_is_builder_frame() ) {
				add_action( 'fusion_builder_before_init', array( $this, 'frontend_load_templates' ) );
				add_action( 'wp_enqueue_scripts', array( $this, 'live_scripts' ), 1001 );
			}

			// Add new setting fields to Fusion Builder.
			add_filter( 'fusion_builder_fields', array( $this, 'add_new_fields' ), 11 );
		}

		/**
		 * Displays admin notice if Fusion Builder is not active.
		 *
		 * @access public
		 * @since 1.0
		 * @return void
		 */
		public function fusion_builder_required_admin_notice() {
			if ( ! class_exists( 'FusionBuilder' ) ) {
				echo '<div class="notice notice-warning is-dismissible">
	             <p>' . esc_attr__( 'Image Hover Add-on for Fusion Builder is installed and activated correctly. However, it will not take any effect until you install and activate Fusion Builder.', 'image-hover' ) . '</p>
	         </div>';
			}
		}

		/**
		 * Loads the plugin language files.
		 *
		 * @access public
		 * @since 1.0
		 * @return void
		 */
		public function textdomain() {

			// Set text domain.
			$domain = 'image-hover';

			// Load the plugin textdomain.
			load_plugin_textdomain( $domain, false, dirname( plugin_basename( FB_IMAGE_HOVER_PLUGIN_FILE ) ) . '/languages/' );
		}

		/**
		 * Enqueue image hover styles on frontend.
		 *
		 * @since 1.0
		 * @access public
		 * @return void
		 */
		public function frontend_styles() {

			// Register styles.
			wp_register_style( 'image-hover-add-on', FB_IMAGE_HOVER_PLUGIN_URL . 'assets/css/min/imagehover-pro.min.css', '', FB_IMAGE_HOVER_VERSION );

			// Enqueue styles on frontend.
			wp_enqueue_style( 'image-hover-add-on' );
		}

		/**
		 * Add templates required for image hover element on front-end.
		 *
		 * @since 1.0
		 * @access public
		 * @return void
		 */
		public function frontend_load_templates() {
			include FB_IMAGE_HOVER_PLUGIN_DIR . 'inc//front-end/image-hover-add-on.php';
		}

		/**
		 * Add js view templates required for image hover element on front-end.
		 *
		 * @since 1.0
		 * @access public
		 * @return void
		 */
		public function live_scripts() {
			wp_enqueue_script( 'image-hover-add-on-view', plugins_url( 'inc/front-end/image-hover-add-on-view.js', __FILE__ ), array(), FB_IMAGE_HOVER_VERSION, true );
		}

		/**
		 * Add new setting fields to Fusion Builder.
		 *
		 * @since 1.0
		 * @access public
		 * @param array $fields The array of fields added with filter.
		 * @return array
		 */
		public function add_new_fields( $fields ) {
			$fields[] = array( 'image_hover_styles', FB_IMAGE_HOVER_PLUGIN_DIR . 'inc/fields/image-hover-styles.php' );
			return $fields;
		}
	}
}

/**
 * Instantiates the Fusion_Builder_Image_Hover class.
 * Make sure the class is properly set-up.
 * The Fusion_Builder_Image_Hover class is a singleton
 * so we can directly access the one true Fusion_Builder_Image_Hover object using this function.
 *
 * @return object Fusion_Builder_Image_Hover
 */
function infi_image_hover_add_on() {
	return Fusion_Builder_Image_Hover::get_instance();
}

/**
 * Instantiate Fusion_Builder_Image_Hover class.
 *
 * @since 1.0
 * @return void
 */
function infi_image_hover_add_on_activate() {
	infi_image_hover_add_on();

	if ( ! class_exists( 'Fusion_Cache' ) && defined( 'FUSION_BUILDER_PLUGIN_DIR' ) ) {
		if ( is_file( FUSION_BUILDER_PLUGIN_DIR . 'inc/lib/inc/class-fusion-cache.php' ) ) {
			include_once FUSION_BUILDER_PLUGIN_DIR . 'inc/lib/inc/class-fusion-cache.php';
		}
	}

	$version = get_option( 'image_hover_add_on_version', false );

	if ( ! $version && class_exists( 'Fusion_Cache' ) ) {
		// Clear cache if no version number found.
		$fusion_cache = new Fusion_Cache();
		$fusion_cache->reset_all_caches();
	} elseif ( version_compare( FB_IMAGE_HOVER_VERSION, $version, '>' ) && class_exists( 'Fusion_Cache' ) ) {

		// Clear cache as the version from database is different than current version.
		$fusion_cache = new Fusion_Cache();
		$fusion_cache->reset_all_caches();
	}
	// Update current version number to database.
	update_option( 'image_hover_add_on_version', FB_IMAGE_HOVER_VERSION );
}
add_action( 'after_setup_theme', 'infi_image_hover_add_on_activate', 11 );

/**
 * Initialize image hover add-on once FB elements are loaded.
 *
 * @since 1.0
 * @return void
 */
function init_image_hover_add_on() {
	require_once FB_IMAGE_HOVER_PLUGIN_DIR . 'inc/image-hover-add-on.php';
}
add_action( 'fusion_builder_shortcodes_init', 'init_image_hover_add_on' );

/**
 * Auto activate elements on plugin activation.
 *
 * @since 1.0
 * @return void
 */
function auto_activate_image_hover_add_on() {

	// Auto activate element.
	if ( function_exists( 'fusion_builder_auto_activate_element' ) ) {
		fusion_builder_auto_activate_element( 'image_hover_add_on' );
	}

	$version = get_option( 'image_hover_add_on_version', false );

	if ( ! $version && class_exists( 'Fusion_Cache' ) ) {

		// Clear cache if no version number found.
		$fusion_cache = new Fusion_Cache();
		$fusion_cache->reset_all_caches();

	} elseif ( version_compare( FB_IMAGE_HOVER_VERSION, $version, '>' ) && class_exists( 'Fusion_Cache' ) ) {

		// Clear cache as the version from database is different than current version.
		$fusion_cache = new Fusion_Cache();
		$fusion_cache->reset_all_caches();
	}

	// Update current version number to database.
	update_option( 'image_hover_add_on_version', FB_IMAGE_HOVER_VERSION );
}

register_activation_hook( FB_IMAGE_HOVER_PLUGIN_FILE, 'auto_activate_image_hover_add_on' );
